/*
 * --COPYRIGHT--,BSD Copyright (c) 2015, Texas Instruments Incorporated
 * All rights reserved. Redistribution and use in source and binary
 * forms, with or without modification, are permitted provided that the
 * following conditions are met: * Redistributions of source code must
 * retain the above copyright notice, this list of conditions and the
 * following disclaimer. * Redistributions in binary form must reproduce
 * the above copyright notice, this list of conditions and the following
 * disclaimer in the documentation and/or other materials provided with
 * the distribution. * Neither the name of Texas Instruments Incorporated 
 * nor the names of its contributors may be used to endorse or promote
 * products derived from this software without specific prior written
 * permission. THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE. --/COPYRIGHT--
 */

/*******************************************************************************
 *
 *  utility.h - Definition file for utility functions
 *  DRV8881EEVM_FIRMWARE
 *  1/30/2015
 *
 ******************************************************************************/

#ifndef UTILITY_H_
#define UTILITY_H_

/*****************************************************************************/

// change to reflect firmware revision
#define FW_VERSION 1.1		// 
#define LED_TIMEOUT 20000	// Count to toggle LED
#define SMCLK_FREQ  16000000	// SMCLK frequency in Hz
#define TIME_32ms	62500	// TimerA interrupt value, (31.25ms or 32
				// updates/s)

// IFS defines
// this are used to calculate the chopping current
#define V3P3_TIMES_GAIN			0.5	// 3.3 divide by 6.6, use
						// constant
#define VREF_BIT				0.0009775	// 1/1023
								// per
								// step
#define TORQUE_STEP				0.25	// 1/4 per step
#define ICHOP_STEP				0.0625	// 1/16 per
							// current step

#define Lower_12_bits			0xFFF

// 

#define	DEFAULT_START_SPEED		200	// initial speed in pulses 
						// per second when
						// starting motor
#define DEFAULT_TARGET_SPEED    1000	// speed of motor once running
#define DEFAULT_ACCEL_RATE		500	// acceleration rate from
						// starting speed to
						// target speed
#define DEFAULT_STOPPING_SPEED  200	// speed to stop motor once
					// deceleration has reached this
					// point
#define DEFAULT_NUM_STEPS		1	// number of steps to move 
						// when manually stepping
#define DEFAULT_STEPS_2_STOP 	1	// number of steps used to stop
					// the motor when manually
					// stepping
#define	DEFAULT_DECAY			6	// Mixed 1tblank / Mixed 1 
						// tblank (use index value 
						// 0 - 8)
#define	DEFAULT_STEP_MODE		3	// 8 rising edge only (use 
						// index value 0 - 8)
#define	DEFAULT_TORQUE			0	// 100% (use index value 0 
						// - 3)
#define	DEFAULT_PWM_OFF_TIME	0	// 20us off (use index value 0 -
					// 2)
#define	DEFAULT_IFS				0
#define	DEFAULT_SENSE_RESISTOR	0.25
#define	DEFAULT_VREF_VAL		512
#define DEFAULT_CURRENT_SETTING 8	// 1A

#define use_one_edge			1	// multiplies all step
						// calculations by 2
											// this 
											// is 
											// legacy 
											// from 
											// the 
											// DRV8846

// defines for motor state machine
#define     IDLE        	1
#define     ACCEL       	2
#define     DECEL       	4
#define     RUN         	8
#define		DECEL2NEWTARGET	16

#define		ACCEL_RATE_COUNT 2000

// GPIO Port 2 Definitions
#define BDECAY		BIT0	// P2.0
#define ADECAY		BIT1	// P2.1
#define ENA			BIT4	// P2.4
#define TOFF		BIT7	// P2.7
#define TRQ1		BIT2	// P2.2
#define TRQ0		BIT5	// P2.5
#define PHA			BIT3	// P2.3
#define ADEC		BIT6	// P2.6

#define CAPD_BDECAY	BIT0
#define CAPD_ADECAY	BIT1
#define CAPD_TRQ1	BIT2
#define CAPD_TRQ0	BIT5
#define CAPD_TOFF	BIT7
#define CAPD_PHA	BIT3
#define CAPD_ADEC	BIT6

// GPIO Port 4 Definitions
#define WAKE		BIT5	// P4.5
#define PHB			BIT2	// P4.2
#define ENB			BIT1	// P4.1
#define FAULT		BIT6	// P4.6
#define V3P3_GPIO   BIT7	// P4.7

// GPIO Port 6 Definitions
#define AVREF		BIT6	// P6.6
#define STATUS		BIT5	// P6.5

// 
// defines to make code more readable, and easier to fix
// 

#define toggle_status_led 		P6OUT ^=  STATUS

#define set_ADEC_hi				P2OUT |=  ADEC
#define set_ADEC_lo				P2OUT &= ~ADEC

#define set_BDECAY_hi			P2OUT |=  BDECAY
#define set_BDECAY_lo			P2OUT &= ~BDECAY
#define set_ADECAY_hi			P2OUT |=  ADECAY
#define set_ADECAY_lo			P2OUT &= ~ADECAY

#define set_PHB_hi				P4OUT |=  PHB
#define set_PHB_lo				P4OUT &= ~PHB
#define set_ENB_hi				P4OUT |=  ENB
#define set_ENB_lo				P4OUT &= ~ENB

#define set_TOFF_hi				P2OUT |=  TOFF
#define set_TOFF_lo				P2OUT &= ~TOFF

#define set_TRQ1_hi				P2OUT |=  TRQ1
#define set_TRQ1_lo				P2OUT &= ~TRQ1
#define set_TRQ0_hi				P2OUT |=  TRQ0
#define set_TRQ0_lo				P2OUT &= ~TRQ0

#define set_ENA_hi				P2OUT |=  ENA
#define set_ENA_lo				P2OUT &= ~ENA
#define set_PHA_hi				P2OUT |=  PHA
#define set_PHA_lo				P2OUT &= ~PHA
#define set_ADEC_hi				P2OUT |=  ADEC
#define set_ADEC_lo				P2OUT &= ~ADEC

#define set_WAKE_hi				P4OUT |=  WAKE
#define set_WAKE_lo				P4OUT &= ~WAKE

#define enable_V3P3_GPIO		P4OUT |= V3P3_GPIO; P4DIR |=  V3P3_GPIO
#define disable_V3P3_GPIO		P4DIR &= ~V3P3_GPIO

// enable/disable output

#define enable_BDECAY_output	P2DIR |=  BDECAY
#define disable_BDECAY_output	P2DIR &= ~BDECAY
#define enable_ADECAY_output	P2DIR |=  ADECAY
#define disable_ADECAY_output	P2DIR &= ~ADECAY

#define disable_BDECAY_input	CAPD  |=  CAPD_BDECAY
#define enable_BDECAY_input		CAPD  &= ~CAPD_BDECAY
#define disable_ADECAY_input	CAPD  |=  CAPD_ADECAY
#define enable_ADECAY_input		CAPD  &= ~CAPD_ADECAY

#define enable_TOFF_output		P2DIR |=  TOFF
#define disable_TOFF_output		P2DIR &= ~TOFF

#define disable_TOFF_input		CAPD  |=  CAPD_TOFF
#define enable_TOFF_input		CAPD  &= ~CAPD_TOFF

#define enable_PHA_output		P2DIR |=  PHA
#define disable_PHA_output		P2DIR &= ~PHA
#define enable_ADEC_output		P2DIR |=  ADEC
#define disable_ADEC_output		P2DIR &= ~ADEC

#define disable_PHA_input		CAPD  |=  CAPD_PHA
#define enable_PHA_input		CAPD  &= ~CAPD_PHA
#define disable_ADEC_input		CAPD  |=  CAPD_ADEC
#define enable_ADEC_input		CAPD  &= ~CAPD_ADEC

#define read_FAULT_pin			(P4IN & FAULT)

#define disable_ENB_output  	P4DIR &= ~ENB
#define enable_ENB_output  	  	P4DIR |= ENB

// Custom Types
typedef enum { false, true } boolean;
typedef enum { low, high } gpio;

/*****************************************************************************/

// Declare Global Variables

// ***********************************************************************************
// Global Variables for GUI Composer *
// ***********************************************************************************

extern boolean  G_ENBL_START_STOP;	// Allow Motor to Move 0 -
					// Disabled 1 - Enabled
extern boolean  G_ENBL_MOVE_STEPS;	// Allow Motor to Move 0 -
					// Disabled 1 - Enabled
extern boolean  G_MOTOR_CMD_ACTIVE1;	// Motor in Motion (Auto) 0 - No
					// Motion 1 - In Motion
extern boolean  G_MOTOR_CMD_ACTIVE2;	// Motor in Motion (Man) 0 - No
					// Motion 1 - In Motion
extern boolean  G_MOTOR_CMD_ACTIVE3;
extern boolean  G_RUN_MOTOR;	// Auto run enabled
extern boolean  G_MANUAL_ADVANCE;	// Manually advance motor
extern boolean  G_ENABLE_MTR_CONTROL;	// message for user to select
					// enable and wake
extern boolean  G_RECIPROCATION;
extern boolean  G_ENBL_RECIPROCATION_BUTTON;
extern boolean  G_IDLE;		// AccelerateState = 1


// GPIO
extern gpio     G_WAKE;		// Wake device 0 - Sleep 1 - Wake
extern gpio     G_DIR;		// Direction 0 - Forward 1 - Reverse
extern gpio     G_ENABLE;	// Enable device 0 - Disabled 1 - Enabled
extern gpio     G_FAULT;	// Fault Status 0 - Fault 1 - OK
extern gpio     G_ADEC;		// Unused Spare 0 - Default 1 - Unused

// PWM Generation
extern unsigned int G_ADECAY;	// Decay Mode (1 of 3)
extern unsigned int G_BDECAY;	// Decay Mode (1 of 3)
extern unsigned int G_ICHOP;	// Chopping current steps (1 of 16)
extern unsigned int G_STEP_MODE;	// Step Mode (1 of 11)
extern unsigned int G_TORQUE;	// Torque (1 of 4)
extern unsigned int G_PWM_OFF_TIME;	// PWM off time (1 of 3)

extern unsigned int G_STARTING_SPEED;	// initial speed in pulses per
					// second when starting motor
extern unsigned int G_TARGET_SPEED;	// speed of motor once running
extern unsigned int G_ACCEL_RATE;	// acceleration rate from starting 
					// speed to target speed
extern unsigned int G_STOPPING_SPEED;	// speed to stop motor once
					// deceleration has reached this
					// point
extern unsigned int G_NUM_STEPS;	// number of steps to move when
					// manually stepping
extern unsigned int G_STEPS_2_STOP;	// number of steps used to stop
					// the motor when manually
					// stepping
extern unsigned long int G_VREF_VAL;	// scale for current

extern float    G_IFS;		// Full scale current
extern float    G_IRSENSE;	// sense value resistor
extern float    G_FIRMWARE_VERSION;	// keep track of firmware

// ///////////////////////
// variables for utility
// ///////////////////////

// variable for the motor state machine
unsigned int    AccelerateState;
unsigned long int SteppingRateTMR;
unsigned long int SteppingRate;
unsigned long int SteppingRateUpdate;
unsigned long int DesiredStepperSpeed;
unsigned long int DesiredTargetSpeed;
unsigned long int StartingSpeed;
unsigned long int StoppingSpeed;
unsigned long int AccelRate;
unsigned long int AccelTimeBase;
unsigned long int AccelerationIncrease;
unsigned long int StepsToStop;
unsigned long int StepsToMove;
unsigned long int tmpStepsToMove;
unsigned long int PriorSteppingRate;

unsigned int    PriorDecayMode;	// return to previous decay mode
											// when 
											// exiting 
											// autodecay
unsigned int    PriorADecay;
unsigned int    PriorBDecay;
boolean         Block_GUI_FAULT_button;
boolean         set_ENB_as_opendrain;
boolean         temp_delay;
boolean         temp_reciprocation;
boolean         ForceUpdate;
boolean         LastCommandWasRunSteps;
boolean         MoveSteps;
boolean         UpdateChoppingCurrent;
boolean         UpdateSteppingRateTMR;
boolean         phase_pins_have_been_set;
gpio            tmp_adec;

float           torque_multiplier;
float           vref_multiplier;
float           step_multiplier;
float           irsense_multiplier;
float           time;
float           time_squared;

unsigned int    last_reciprocation_state;	// reciprocation flag, =1
						// last action is
						// reciprocation

unsigned int    LED_TOGGLE_CNT;	// count to determine if LED should be
				// toggled
unsigned long int Steps2Target;	// number of steps to reach target speed
										// from 
										// starting 
										// speed 
										// using 
										// acceleration 
										// rate
unsigned long int Steps2Stop;	// number of steps to stop
										// from 
										// target 
										// speed 
										// using 
										// acceleration 
										// rate
unsigned long int Steps2Decel;	// step number to change from accel to
				// decel
										// in 
										// manual 
										// mode
unsigned int    delay_cycles;
unsigned int    move_to_next_step;
unsigned int    step_offset;

unsigned int    current_scale;
unsigned int    Next_a_step;
unsigned int    Next_b_step;
unsigned int    Next_a_sine_table;
unsigned int    Next_b_sine_table;
unsigned int    Next_a_value;
unsigned int    Next_b_value;
unsigned int    Next_eighth_a;
unsigned int    Next_eighth_b;
unsigned int    Next_full_a;
unsigned int    Next_full_b;
unsigned int    Next_half_a;
unsigned int    Next_half_b;
unsigned int    Next_quarter_a;
unsigned int    Next_quarter_b;
unsigned int    Next_sixteenth_a;
unsigned int    Next_sixteenth_b;
unsigned int    Next_Non_CirA;
unsigned int    Next_Non_CirB;
unsigned int    Next_PHA;
unsigned int    Next_PHB;
unsigned int    Next_value_calculated;

unsigned int    tmp_gdir;
unsigned int    tmp_ichop_val;
unsigned int    temp_step_mode;
unsigned int    temp_g_enable;



/*****************************************************************************/

// Function Declarations

// Main Functions
void            Initialize();
void            UpdateDeviceControls();
void            UpdateGUIControls();
void            UpdateRegisters();
void            UpdateSpeed();

// GPIO/GUI Functions
void            SetDefaultGUIVals();
void            SetDefaultGPIOVals();

// Motor Functions
void            DetermineMotorMovement();
void            MotorStateMachine();
void            CalculateNextSineValue();

#endif				/* UTILITY_H_ */
